function [flexure_properties]=FlexureResonantResponse(filename,Flexure_type,varargin)
%% Determine properties of the flexure web based on the 
% time domain resonant frquency.
%  Returns a structure containing the properties of a flexure, based on
%  time domain measurements of end flag oscillation.
%  arg1 = filename of data captured using the TDS2012 scope
%  Flexure_type = 'top-long' 'top-short' 'bottom-long' 'bottom-short' indicating the appropriate end type
%  verbose = [true] or false, boolean for verbose output (including plots)
%  freq = guess of nominal resonant frequency
%  I = moment of inertia for the end about the flexure
%
% Graham Allen
% 8/19/2009

%% Time domain data

%filename='Flexure2S.CSV';
%Flexure_type='long';
if (nargin<3)
    verbose=true;
else 
    verbose=varargin{1};
end
data=ReadTDS2012(filename);

% Set start and stop time for the ringdown data
t_start=0.1;
t_stop=max(data(:,1)); %Use all data
fit_seg=find(data(:,1)>t_start & data(:,1)<t_stop);

%Basic properties
INCH=0.0254;
Ebecu=125e9; %Pa
b=0.25*INCH; %Flexure height in m
ax=0.5/2*INCH; %Flexure radius in m

ringdown=[];
switch Flexure_type
    case 'bottom-long'
        ringdown(2)=87.1;
        Ixx=4.91823e-7; %kg*m^2
    case 'bottom-short'
        ringdown(2)=110;
        Ixx=3.00628e-7; %kg*m^2
    case 'top-long'
        ringdown(2)=87;
        Ixx=3.40503e-7; %kg*m^2
    case 'top-short'
        ringdown(2)=110;
        Ixx=2.6285e-7; %kg*m^2
    case 'custom'
        Ixx=varargin{3};
        ringdown(2)=varargin{2};
end
% Set initial parameters for fit
ringdown(1) = 0.2; % Initial amplitude
ringdown(3) = 2; % Sine phase
ringdown(4) = 50; % Q
ringdown(5) = .05;

% Define a damped sinusoid. 
damped_sine = @(p,t)  p(1) * sin(2*pi*p(2)*t + p(3)) .* exp(-(t*2*pi*p(2))/(2*p(4)) )+p(5);

%Plot the initial guess if needed
%plot(data(fit_seg,1),data(fit_seg,2),data(fit_seg,1),damped_sine(ringdown,viton_clamp(fit_seg,1)))

%Perform non-linear fit to the time domain data;
fit_ringdown=nlinfit(data(fit_seg,1),data(fit_seg,2),damped_sine,ringdown);

%plot(viton_clamp(fit_seg,1),viton_clamp(fit_seg,2),viton_clamp(fit_seg,1),damped_sine(fit_ringdown,viton_clamp(fit_seg,1)),viton_clamp(fit_seg,1),damped_sine(ringdown,viton_clamp(fit_seg,1)))
figure;
subplot(2,1,1)
plot(data(:,1),data(:,2),data(fit_seg,1),damped_sine(fit_ringdown,data(fit_seg,1)));
xlabel('Time (s)')
ylabel('Voltage (V)')
axis tight
tt={};
tt{1}=filename;
tt{2}=['f=',num2str(fit_ringdown(2),4), ' Hz'];
tt{3}=['Q=',num2str(fit_ringdown(4),3)];
text(0.025,0.045,tt','FontSize',18)
IDplot;

subplot(2,1,2)
residuals = (data(fit_seg,2)) - damped_sine(fit_ringdown,data(fit_seg,1));
plot(data(fit_seg,1),residuals);
xlabel('Time (s)')
ylabel('Residual Voltage (V_{meas}-V_{fit})');
IDplot;

flexure_properties=struct;
flexure_properties.freq=fit_ringdown(2);
flexure_properties.Q=fit_ringdown(4);
flexure_properties.type=Flexure_type;
flexure_properties.filename=filename;
flexure_properties.stiffness=(2*pi*flexure_properties.freq)^2*Ixx; %kg*m^2*s^-2=J
flexure_properties.thick=( (9*pi*sqrt(ax)*flexure_properties.stiffness)/(2*Ebecu*b) )^(2/5);
flexure_properties.Ixx=Ixx;
flexure_properties
