function [flexure_properties]=FlexureResonantResponse(filename,Flexure_type,varargin)
%% Determine properties of the flexure web based on the 
% time domain resonant frquency.
%  Returns a structure containing the properties of a flexure, based on
%  time domain measurements of end flag oscillation.
%  arg1 = filename of data captured using the TDS2012 scope
%  Flexure_type = 'top-long' 'top-short' 'bottom-long' 'bottom-short' indicating the appropriate end type
%  label = Label to attach to the flexure, if empty defaults to filename
%  verbose = [true] or false, boolean for verbose output (including plots)
%  freq = guess of nominal resonant frequency
%  I = moment of inertia for the end about the flexure
%
% Graham Allen
% 8/19/2009

%% Time domain data

MIN_STIFF=0.086/0.97; %  3 percent error margin
MAX_STIFF=0.193*0.97; 

%filename='Flexure2S.CSV';
%Flexure_type='long';
if (nargin<3)
    label=[];
else 
    label=varargin{1};
end
if isempty(label)
    label=filename;
end

if (nargin<4)
    verbose=true;
else 
    verbose=varargin{2};
end
data=ReadTDS2012(filename);

dirc=fileparts(filename);

% Set start and stop time for the ringdown data
%t_start=0.1;
t_start=min(data(:,1));
t_stop=max(data(:,1)); %Use all data
fit_seg=find(data(:,1)>t_start & data(:,1)<t_stop);
Fs=1/(data(2,1)-data(1,1));

%Basic properties
INCH=0.0254;
Ebecu=125e9; %Pa
b=0.25*INCH; %Flexure height in m
ax=0.5/2*INCH; %Flexure radius in m

ringdown=[];
switch Flexure_type
    case 'bottom-long'
        %ringdown(2)=87.1;
        Ixx=4.902e-7; %kg*m^2
    case 'bottom-short'
        %ringdown(2)=105;
        Ixx=3.00628e-7; %kg*m^2
    case 'top-long'
        %ringdown(2)=87;
        Ixx=3.390e-7; %kg*m^2
    case 'top-short'
        %ringdown(2)=110;
        Ixx=2.6285e-7; %kg*m^2
    case 'custom'
        %Ixx=varargin{3};
        ringdown(2)=varargin{2};
end

options=statset('Robust','on');
% Define a damped sinusoid. 
damped_sine = @(p,t)  p(1) * sin(2*pi*p(2)*t + p(3)) .* exp(-(t*2*pi*p(2))/(2*p(4)) )+p(5);

%Plot the initial guess if needed
%plot(data(fit_seg,1),data(fit_seg,2),data(fit_seg,1),damped_sine(ringdown,viton_clamp(fit_seg,1)))

%Perform non-linear fit to the time domain data;

% Set initial parameters for fit
ringdown(1) = (max(data(fit_seg,2))-min(data(fit_seg,2)))/2; % Initial amplitude

%Use fft to guess frequency
NFFT = 2^nextpow2(length(fit_seg)); % Next power of 2 from length of y
Y = abs(fft(data(fit_seg,2).*HANNING(length(fit_seg)),NFFT)/length(fit_seg)).^2;
f = Fs/2*linspace(0,1,NFFT/2);
[val,indx]=max(Y);
ringdown(2)=f(indx);

%Zero-crossing to guess phase
[m,indx]=min(abs(data(fit_seg,2)));
t_min=data(fit_seg(indx),1);
ringdown(3)=-2*pi*ringdown(1)*t_min;  % Sine phase

%Other parameters
ringdown(4) = 50; % Q
ringdown(5) = mean(data(fit_seg,2)); %DC offset

fit_ringdown=nlinfit(data(fit_seg,1),data(fit_seg,2),damped_sine,ringdown,options);

flexure_properties=struct;
flexure_properties.label=label;
flexure_properties.freq=fit_ringdown(2);
flexure_properties.Q=fit_ringdown(4);
flexure_properties.type=Flexure_type;
flexure_properties.filename=filename;
flexure_properties.stiffness=(2*pi*flexure_properties.freq)^2*Ixx; %kg*m^2*s^-2=J
flexure_properties.thick=( (9*pi*sqrt(ax)*flexure_properties.stiffness)/(2*Ebecu*b) )^(2/5);
flexure_properties.Ixx=Ixx;

if (flexure_properties.stiffness<=MAX_STIFF & flexure_properties.stiffness>=MIN_STIFF)
    result='PASS';
else
    result='FAIL';
end

%plot(viton_clamp(fit_seg,1),viton_clamp(fit_seg,2),viton_clamp(fit_seg,1),damped_sine(fit_ringdown,viton_clamp(fit_seg,1)),viton_clamp(fit_seg,1),damped_sine(ringdown,viton_clamp(fit_seg,1)))
figure;
%subplot(2,1,1)
plot(data(:,1),data(:,2),data(fit_seg,1),damped_sine(fit_ringdown,data(fit_seg,1)));
xlabel('Time (s)')
ylabel('Voltage (V)')
axis tight
tt={};
title([label, ' -- ', Flexure_type, ' -- ', result]);
tt{1}=label;
tt{2}=['f=',num2str(fit_ringdown(2),4), ' Hz'];
tt{3}=['Q=',num2str(fit_ringdown(4),3)];
tt{4}=['Stiffness = ', num2str(flexure_properties.stiffness,3), 'N\cdotm'];
tt{5}=Flexure_type;
text(0.025,0.045,tt','FontSize',18)
IDplot;

%orient landscape;
plot_file=[dirc,label,'-',Flexure_type];
print('-djpeg',plot_file)
% subplot(2,1,2)
% residuals = (data(fit_seg,2)) - damped_sine(fit_ringdown,data(fit_seg,1));
% plot(data(fit_seg,1),residuals);
% xlabel('Time (s)')
% ylabel('Residual Voltage (V_{meas}-V_{fit})');
% IDplot;



%flexure_properties
